#!/bin/bash

# Конфигурация Transmission
TRANSMISSION_HOST="localhost"
TRANSMISSION_PORT="9091"
TRANSMISSION_USER="*********"
TRANSMISSION_PASSWORD="*********"

# Список трекеров по умолчанию
DEFAULT_TRACKERS=(
    "udp://opentor.net:6969"
    "udp://exodus.desync.com:6969/announce"
)

# Файл с дополнительными трекерами (опционально)
TRACKER_FILE="$HOME/.config/torrent_trackers.txt"

# Загружаем трекеры из файла, если он существует
load_trackers_from_file() {
    local trackers=()
    if [ -f "$TRACKER_FILE" ]; then
        echo "  Загружаю трекеры из файла: $TRACKER_FILE"
        while IFS= read -r line || [ -n "$line" ]; do
            # Удаляем пробелы в начале и конце
            line=$(echo "$line" | sed 's/^[[:space:]]*//;s/[[:space:]]*$//')
            # Пропускаем пустые строки и комментарии
            if [[ -n "$line" && ! "$line" =~ ^[[:space:]]*# ]]; then
                trackers+=("$line")
                echo "    ✓ $line"
            fi
        done < "$TRACKER_FILE"
        echo ""
    else
        echo "  Файл с трекерами не найден: $TRACKER_FILE"
        echo ""
    fi
    # Возвращаем массив трекеров
    if [ ${#trackers[@]} -gt 0 ]; then
        printf '%s\n' "${trackers[@]}"
    fi
}

# Загружаем трекеры
echo "🔧 Загрузка списка трекеров..."
echo "Трекеры по умолчанию:"
for tracker in "${DEFAULT_TRACKERS[@]}"; do
    echo "  ✓ $tracker"
done

echo ""

# Загружаем трекеры из файла
FILE_TRACKERS_STRING=$(load_trackers_from_file)

# Преобразуем строку в массив, только если есть данные
FILE_TRACKERS=()
if [ -n "$FILE_TRACKERS_STRING" ]; then
    while IFS= read -r line; do
        # Проверяем, что строка не пустая и является валидным URL трекера
        if [[ -n "$line" && "$line" =~ ^(udp|http|https):// ]]; then
            FILE_TRACKERS+=("$line")
        fi
    done <<< "$FILE_TRACKERS_STRING"
fi

# Объединяем трекеры
TRACKERS=("${DEFAULT_TRACKERS[@]}")
if [ ${#FILE_TRACKERS[@]} -gt 0 ]; then
    TRACKERS=("${DEFAULT_TRACKERS[@]}" "${FILE_TRACKERS[@]}")
fi

# Проверяем, есть ли вообще трекеры
if [ ${#TRACKERS[@]} -eq 0 ]; then
    echo "❌ Ошибка: Не указано ни одного трекера"
    exit 1
fi

echo "📋 Итоговый список трекеров (${#TRACKERS[@]} шт.):"
for ((i=0; i<${#TRACKERS[@]}; i++)); do
    echo "  $((i+1)). ${TRACKERS[i]}"
done
echo ""

# Функция для формирования строки с трекерами
build_tracker_args() {
    local tracker_args=""
    for tracker in "${TRACKERS[@]}"; do
        tracker_args+=" --tracker \"$tracker\""
    done
    echo "$tracker_args"
}

# Парсим аргументы командной строки
show_help() {
    echo "Использование: $0 [OPTIONS] /путь/к/директории|/путь/к/файлу"
    echo ""
    echo "Options:"
    echo "  -u, --user USERNAME      Имя пользователя transmission"
    echo "  -p, --password PASSWORD  Пароль transmission"
    echo "  -h, --host HOST          Хост transmission (по умолчанию: localhost)"
    echo "  -P, --port PORT          Порт transmission (по умолчанию: 9091)"
    echo "  -t, --tracker-file FILE  Файл с трекерами (по умолчанию: \$HOME/.config/torrent_trackers.txt)"
    echo "  -o, --output DIR         Директория для сохранения торрентов (по умолчанию: \$HOME/torrents)"
    echo "  -c, --comment COMMENT    Комментарий для торрента"
    echo "  -w, --webseed URL        Webseed URL (можно использовать несколько раз)"
    echo "  -a, --anonymize          Анонимный режим (не включать личную информацию)"
    echo "  --no-add                 Только создать торренты, не добавлять в transmission"
    echo "  --dir-torrent            Создать один торрент для всей директории (только для директорий)"
    echo "  --name NAME              Имя для торрента директории (только с --dir-torrent)"
    echo "  --help                   Показать эту справку"
    echo ""
    echo "Примеры:"
    echo "  $0 /media/sdd/files/media/Video/Artem"
    echo "  $0 --dir-torrent --name \"My Collection\" /path/to/dir"
    echo "  $0 -c \"High quality video\" -a /path/to/file.mkv"
    echo "  $0 -w http://cdn.example.com/files/ -w http://mirror.example.com/files/ /path/to/dir"
    echo "  $0 --comment \"2024 Release\" --webseed \"http://seed.example.com/\" --anonymize /path/to/file"
    echo ""
    echo "Формат файла с трекерами:"
    echo "  # Комментарии начинаются с #"
    echo "  udp://tracker1.com:6969"
    echo "  http://tracker2.com/announce"
    echo "  # Пустые строки игнорируются"
    exit 0
}

# Парсим аргументы
TARGET_PATH=""
TORRENT_DIR="$HOME/torrents"
ADD_TO_TRANSMISSION=true
DIR_TORRENT=false
TORRENT_NAME=""
TORRENT_COMMENT=""
WEBSEEDS=()
ANONYMIZE=false

while [[ $# -gt 0 ]]; do
    case $1 in
        -u|--user)
            TRANSMISSION_USER="$2"
            shift 2
            ;;
        -p|--password)
            TRANSMISSION_PASSWORD="$2"
            shift 2
            ;;
        -h|--host)
            TRANSMISSION_HOST="$2"
            shift 2
            ;;
        -P|--port)
            TRANSMISSION_PORT="$2"
            shift 2
            ;;
        -t|--tracker-file)
            TRACKER_FILE="$2"
            echo "🔧 Загрузка трекеров из файла: $TRACKER_FILE"
            # Перезагружаем трекеры из указанного файла
            FILE_TRACKERS_STRING=$(load_trackers_from_file)
            FILE_TRACKERS=()
            if [ -n "$FILE_TRACKERS_STRING" ]; then
                while IFS= read -r line; do
                    # Проверяем, что строка не пустая и является валидным URL трекера
                    if [[ -n "$line" && "$line" =~ ^(udp|http|https):// ]]; then
                        FILE_TRACKERS+=("$line")
                    fi
                done <<< "$FILE_TRACKERS_STRING"
            fi
            # Обновляем список трекеров
            TRACKERS=("${DEFAULT_TRACKERS[@]}")
            if [ ${#FILE_TRACKERS[@]} -gt 0 ]; then
                TRACKERS=("${DEFAULT_TRACKERS[@]}" "${FILE_TRACKERS[@]}")
            fi
            echo "📋 Обновленный список трекеров (${#TRACKERS[@]} шт.):"
            for ((i=0; i<${#TRACKERS[@]}; i++)); do
                echo "  $((i+1)). ${TRACKERS[i]}"
            done
            echo ""
            shift 2
            ;;
        -o|--output)
            TORRENT_DIR="$2"
            shift 2
            ;;
        -c|--comment)
            TORRENT_COMMENT="$2"
            shift 2
            ;;
        -w|--webseed)
            WEBSEEDS+=("$2")
            shift 2
            ;;
        -a|--anonymize)
            ANONYMIZE=true
            shift
            ;;
        --no-add)
            ADD_TO_TRANSMISSION=false
            shift
            ;;
        --dir-torrent)
            DIR_TORRENT=true
            shift
            ;;
        --name)
            TORRENT_NAME="$2"
            shift 2
            ;;
        --help)
            show_help
            ;;
        -*)
            echo "❌ Неизвестная опция: $1"
            echo "Используйте --help для справки"
            exit 1
            ;;
        *)
            TARGET_PATH="$1"
            shift
            ;;
    esac
done

if [ -z "$TARGET_PATH" ]; then
    echo "❌ Ошибка: Не указан путь к файлу или директории"
    echo "Используйте: $0 [OPTIONS] /путь/к/директории|/путь/к/файлу"
    echo "Используйте --help для подробной справки"
    exit 1
fi

# Проверяем существование пути
if [ ! -e "$TARGET_PATH" ]; then
    echo "❌ Ошибка: Путь '$TARGET_PATH' не существует"
    exit 1
fi

# Определяем тип пути (файл или директория)
if [ -f "$TARGET_PATH" ]; then
    TARGET_TYPE="file"
    TARGET_DIR=$(dirname "$TARGET_PATH")
    TARGET_NAME=$(basename "$TARGET_PATH")
    
    # Проверяем, что для файла не используется --dir-torrent
    if [ "$DIR_TORRENT" = true ]; then
        echo "❌ Ошибка: Опция --dir-torrent может использоваться только для директорий"
        exit 1
    fi
    
    echo "🎯 Обрабатываем файл: $TARGET_NAME"
    echo "📁 Директория файла: $TARGET_DIR"
    
elif [ -d "$TARGET_PATH" ]; then
    TARGET_TYPE="directory"
    TARGET_DIR="$TARGET_PATH"
    
    if [ "$DIR_TORRENT" = true ]; then
        echo "📦 Обрабатываем директорию как единый торрент: $TARGET_DIR"
        if [ -n "$TORRENT_NAME" ]; then
            echo "🏷️  Имя торрента: $TORRENT_NAME"
        fi
    else
        echo "📁 Обрабатываем директорию (каждый файл отдельно): $TARGET_DIR"
    fi
else
    echo "❌ Ошибка: '$TARGET_PATH' не является файлом или директорией"
    exit 1
fi

# Если имя торрента не указано, генерируем его из имени директории
if [ "$DIR_TORRENT" = true ] && [ -z "$TORRENT_NAME" ]; then
    TORRENT_NAME=$(basename "$TARGET_DIR")
    echo "🏷️  Автоматическое имя торрента: $TORRENT_NAME"
fi

# Показываем дополнительные опции
if [ -n "$TORRENT_COMMENT" ]; then
    echo "💬 Комментарий торрента: $TORRENT_COMMENT"
fi

if [ ${#WEBSEEDS[@]} -gt 0 ]; then
    echo "🌐 Webseed URLs (${#WEBSEEDS[@]} шт.):"
    for ((i=0; i<${#WEBSEEDS[@]}; i++)); do
        echo "  $((i+1)). ${WEBSEEDS[i]}"
    done
fi

if [ "$ANONYMIZE" = true ]; then
    echo "🎭 Анонимный режим: включен"
fi

# Проверяем, установлен ли transmission-cli
if ! command -v transmission-create &> /dev/null; then
    echo "❌ Ошибка: transmission-cli не установлен"
    echo "Установите: sudo apt install transmission-cli"
    exit 1
fi

# Формируем аргументы трекеров
TRACKER_ARGS=$(build_tracker_args)

# Функция для формирования дополнительных аргументов transmission-create
build_extra_args() {
    local extra_args=""
    
    # Комментарий
    if [ -n "$TORRENT_COMMENT" ]; then
        extra_args+=" --comment \"$TORRENT_COMMENT\""
    fi
    
    # Webseeds
    for webseed in "${WEBSEEDS[@]}"; do
        extra_args+=" --webseed \"$webseed\""
    done
    
    # Анонимный режим
    if [ "$ANONYMIZE" = true ]; then
        extra_args+=" --anonymize"
    fi
    
    echo "$extra_args"
}

# Формируем дополнительные аргументы
EXTRA_ARGS=$(build_extra_args)

# Функция для выполнения команд transmission-remote с аутентификацией
transmission_cmd() {
    local cmd="$1"
    
    if [ -n "$TRANSMISSION_USER" ] && [ -n "$TRANSMISSION_PASSWORD" ]; then
        transmission-remote "$TRANSMISSION_HOST:$TRANSMISSION_PORT" \
            --auth "$TRANSMISSION_USER:$TRANSMISSION_PASSWORD" \
            $cmd
    else
        transmission-remote "$TRANSMISSION_HOST:$TRANSMISSION_PORT" \
            $cmd
    fi
}

if [ "$ADD_TO_TRANSMISSION" = true ]; then
    # Проверяем, доступен ли transmission-remote и демон
    echo "🔌 Проверяем подключение к transmission-daemon..."
    if ! transmission_cmd "-l" &> /dev/null; then
        echo "❌ Ошибка: Не удалось подключиться к transmission-daemon"
        echo "Проверьте:"
        echo "  1. Запущен ли демон: transmission-daemon"
        echo "  2. Правильность хоста и порта: $TRANSMISSION_HOST:$TRANSMISSION_PORT"
        if [ -n "$TRANSMISSION_USER" ]; then
            echo "  3. Правильность учетных данных: $TRANSMISSION_USER:******"
        fi
        echo "  4. Настройки аутентификации в /etc/transmission-daemon/settings.json"
        exit 1
    fi

    echo "✅ Успешное подключение к transmission-daemon"
else
    echo "🔕 Режим без добавления в transmission (--no-add)"
fi

echo ""

# Создаем папку для торрентов, если не существует
mkdir -p "$TORRENT_DIR"

echo "💾 Торренты будут сохранены в: $TORRENT_DIR"
if [ "$ADD_TO_TRANSMISSION" = true ]; then
    echo "🌐 Transmission: $TRANSMISSION_HOST:$TRANSMISSION_PORT"
    if [ -n "$TRANSMISSION_USER" ]; then
        echo "👤 Пользователь: $TRANSMISSION_USER"
    fi
fi
echo ""

# Функция для добавления торрента в transmission с правильным путем
add_torrent_to_transmission() {
    local torrent_file="$1"
    local download_dir="$2"
    local torrent_name="$3"
    
    echo "  📤 Добавляю торрент в transmission: $(basename "$torrent_file")"
    echo "  📁 Путь к файлам: $download_dir"
    if [ -n "$torrent_name" ]; then
        echo "  🏷️  Имя торрента: $torrent_name"
    fi
    
    # Добавляем торрент в transmission с указанием правильной директории
    local add_result
    if [ -n "$TRANSMISSION_USER" ] && [ -n "$TRANSMISSION_PASSWORD" ]; then
        add_result=$(transmission-remote "$TRANSMISSION_HOST:$TRANSMISSION_PORT" \
            --auth "$TRANSMISSION_USER:$TRANSMISSION_PASSWORD" \
            --add "$torrent_file" --download-dir "$download_dir" 2>&1)
    else
        add_result=$(transmission-remote "$TRANSMISSION_HOST:$TRANSMISSION_PORT" \
            --add "$torrent_file" --download-dir "$download_dir" 2>&1)
    fi
    
    if [ $? -eq 0 ]; then
        echo "  ✅ Торрент добавлен в transmission с путем: $download_dir"
        
        # Проверяем статус добавленного торрента
        check_torrent_status "$torrent_name"
        return 0
    else
        echo "  ❌ Ошибка при добавлении торрента: $add_result"
        return 1
    fi
}

# Функция для проверки статуса торрента в transmission
check_torrent_status() {
    local torrent_name="$1"
    
    # Ждем немного, чтобы торрент успел появиться в списке
    sleep 1
    
    # Ищем торрент по имени в списке transmission
    local torrent_info
    if [ -n "$TRANSMISSION_USER" ] && [ -n "$TRANSMISSION_PASSWORD" ]; then
        torrent_info=$(transmission-remote "$TRANSMISSION_HOST:$TRANSMISSION_PORT" \
            --auth "$TRANSMISSION_USER:$TRANSMISSION_PASSWORD" \
            -l | grep "$torrent_name")
    else
        torrent_info=$(transmission-remote "$TRANSMISSION_HOST:$TRANSMISSION_PORT" \
            -l | grep "$torrent_name")
    fi
    
    if [ -n "$torrent_info" ]; then
        echo "  ✅ Торрент найден в списке transmission"
        
        # Получаем подробную информацию о торренте
        local torrent_id
        torrent_id=$(echo "$torrent_info" | awk '{print $1}')
        if [ -n "$torrent_id" ] && [ "$torrent_id" -eq "$torrent_id" ] 2>/dev/null; then
            echo "  📊 Статус:"
            if [ -n "$TRANSMISSION_USER" ] && [ -n "$TRANSMISSION_PASSWORD" ]; then
                transmission-remote "$TRANSMISSION_HOST:$TRANSMISSION_PORT" \
                    --auth "$TRANSMISSION_USER:$TRANSMISSION_PASSWORD" \
                    -t "$torrent_id" -i | grep -E '(Name|Location|Percent|Status|Ratio|Downloaded|Uploaded)' | head -6 | while read -r line; do
                    echo "    $line"
                done
            else
                transmission-remote "$TRANSMISSION_HOST:$TRANSMISSION_PORT" \
                    -t "$torrent_id" -i | grep -E '(Name|Location|Percent|Status|Ratio|Downloaded|Uploaded)' | head -6 | while read -r line; do
                    echo "    $line"
                done
            fi
        fi
    else
        echo "  ⚠️  Торрент не найден в списке transmission"
    fi
}

# Функция для проверки, существует ли уже торрент в transmission
is_torrent_in_transmission() {
    local torrent_name="$1"
    
    if [ -n "$TRANSMISSION_USER" ] && [ -n "$TRANSMISSION_PASSWORD" ]; then
        transmission-remote "$TRANSMISSION_HOST:$TRANSMISSION_PORT" \
            --auth "$TRANSMISSION_USER:$TRANSMISSION_PASSWORD" \
            -l | grep -q "$torrent_name"
    else
        transmission-remote "$TRANSMISSION_HOST:$TRANSMISSION_PORT" \
            -l | grep -q "$torrent_name"
    fi
    return $?
}

# Функция для создания торрент-файла для одного файла (исправленная)
create_single_torrent_file() {
    local input_file="$1"
    local output_file="$2"
    
    echo "  🔨 Создаю торрент: $(basename "$output_file")"
    
    # Собираем команду в массив для безопасной обработки пробелов
    local cmd_args=("transmission-create" "-o" "$output_file")
    
    # Добавляем трекеры
    for tracker in "${TRACKERS[@]}"; do
        cmd_args+=("--tracker" "$tracker")
    done
    
    # Добавляем комментарий
    if [ -n "$TORRENT_COMMENT" ]; then
        cmd_args+=("--comment" "$TORRENT_COMMENT")
    fi
    
    # Добавляем webseeds
    for webseed in "${WEBSEEDS[@]}"; do
        cmd_args+=("--webseed" "$webseed")
    done
    
    # Добавляем анонимный режим
    if [ "$ANONYMIZE" = true ]; then
        cmd_args+=("--anonymize")
    fi
    
    # Добавляем входной файл (экранируем пробелы)
    cmd_args+=("$input_file")
    
    # Выполняем команду
    "${cmd_args[@]}"
    return $?
}

# Функция для создания торрент-файла для директории (исправленная)
create_dir_torrent_file() {
    local input_dir="$1"
    local output_file="$2"
    
    echo "  🔨 Создаю торрент для директории: $(basename "$output_file")"
    echo "  📁 Содержимое директории: $input_dir"
    
    # Показываем содержимое директории
    local file_count=$(find "$input_dir" -type f | wc -l)
    local dir_count=$(find "$input_dir" -type d | wc -l)
    echo "  📊 Файлов: $file_count, Папок: $((dir_count - 1))"
    
    # Переходим в родительскую директорию и создаем торрент для папки
    local parent_dir=$(dirname "$input_dir")
    local dir_name=$(basename "$input_dir")
    
    echo "  📂 Создаю торрент из директории: $parent_dir"
    echo "  📁 Для папки: $dir_name"
    
    # Собираем команду в массив для безопасной обработки пробелов
    local cmd_args=("transmission-create" "-o" "$output_file")
    
    # Добавляем трекеры
    for tracker in "${TRACKERS[@]}"; do
        cmd_args+=("--tracker" "$tracker")
    done
    
    # Добавляем комментарий
    if [ -n "$TORRENT_COMMENT" ]; then
        cmd_args+=("--comment" "$TORRENT_COMMENT")
    fi
    
    # Добавляем webseeds
    for webseed in "${WEBSEEDS[@]}"; do
        cmd_args+=("--webseed" "$webseed")
    done
    
    # Добавляем анонимный режим
    if [ "$ANONYMIZE" = true ]; then
        cmd_args+=("--anonymize")
    fi
    
    # Добавляем имя директории
    cmd_args+=("$dir_name")
    
    # Создаем торрент для директории, находясь в родительской директории
    cd "$parent_dir" && "${cmd_args[@]}"
    local result=$?
    cd - > /dev/null
    
    return $result
}

# Функция для обработки одного файла
process_single_file() {
    local file="$1"
    local filename=$(basename "$file")
    local torrent_name="$TORRENT_DIR/${filename}.torrent"
    
    if [ "$ADD_TO_TRANSMISSION" = true ]; then
        # Проверяем, существует ли уже торрент в transmission
        if is_torrent_in_transmission "$filename"; then
            echo "⏭️  Пропускаю (уже в transmission): $filename"
            return 1
        fi
    fi
    
    # Если торрент уже существует
    if [ -f "$torrent_name" ]; then
        echo "📁 Торрент существует: $filename"
        if [ "$ADD_TO_TRANSMISSION" = true ]; then
            echo "  📤 Добавляю существующий торрент в transmission"
            if add_torrent_to_transmission "$torrent_name" "$(dirname "$file")" "$filename"; then
                return 0
            else
                return 1
            fi
        else
            echo "  ⏭️  Пропускаю добавление (режим --no-add)"
            return 1
        fi
    fi
    
    echo "🔨 Создаю торрент для: $filename"
    
    # Создаем торрент
    if create_single_torrent_file "$file" "$torrent_name"; then
        echo "  ✅ Торрент создан: $(basename "$torrent_name")"
        
        # Добавляем торрент в transmission, если не отключено
        if [ "$ADD_TO_TRANSMISSION" = true ]; then
            if add_torrent_to_transmission "$torrent_name" "$(dirname "$file")" "$filename"; then
                return 0
            else
                return 1
            fi
        else
            echo "  💾 Торрент сохранен, добавление в transmission отключено"
            return 0
        fi
    else
        echo "  ❌ Ошибка при создании торрента для: $filename"
        return 1
    fi
}

# Функция для обработки директории как единого торрента
process_directory_as_torrent() {
    local directory="$1"
    local torrent_name="$TORRENT_DIR/${TORRENT_NAME}.torrent"
    
    if [ "$ADD_TO_TRANSMISSION" = true ]; then
        # Проверяем, существует ли уже торрент в transmission
        if is_torrent_in_transmission "$TORRENT_NAME"; then
            echo "⏭️  Пропускаю (уже в transmission): $TORRENT_NAME"
            return 1
        fi
    fi
    
    # Если торрент уже существует
    if [ -f "$torrent_name" ]; then
        echo "📁 Торрент для директории существует: $TORRENT_NAME"
        if [ "$ADD_TO_TRANSMISSION" = true ]; then
            echo "  📤 Добавляю существующий торрент в transmission"
            if add_torrent_to_transmission "$torrent_name" "$(dirname "$directory")" "$TORRENT_NAME"; then
                return 0
            else
                return 1
            fi
        else
            echo "  ⏭️  Пропускаю добавление (режим --no-add)"
            return 1
        fi
    fi
    
    echo "🔨 Создаю торрент для директории: $TORRENT_NAME"
    
    # Создаем торрент для всей директории
    if create_dir_torrent_file "$directory" "$torrent_name"; then
        echo "  ✅ Торрент для директории создан: $(basename "$torrent_name")"
        
        # Добавляем торрент в transmission, если не отключено
        if [ "$ADD_TO_TRANSMISSION" = true ]; then
            # Для директорного торрента используем родительскую директорию как download-dir
            if add_torrent_to_transmission "$torrent_name" "$(dirname "$directory")" "$TORRENT_NAME"; then
                return 0
            else
                return 1
            fi
        else
            echo "  💾 Торрент сохранен, добавление в transmission отключено"
            return 0
        fi
    else
        echo "  ❌ Ошибка при создании торрента для директории: $TORRENT_NAME"
        return 1
    fi
}

# Функция для обработки директории (каждый файл отдельно)
process_directory_files() {
    local directory="$1"
    
    echo "🔍 Поиск файлов в директории: $directory"
    local file_count=$(find "$directory" -maxdepth 1 -type f | wc -l)
    echo "📊 Найдено файлов: $file_count"
    echo ""
    
    local created=0
    local added=0
    local skipped=0
    
    # Обрабатываем каждый файл в директории (с поддержкой пробелов в именах)
    while IFS= read -r -d '' file; do
        if process_single_file "$file"; then
            if [ "$ADD_TO_TRANSMISSION" = true ]; then
                ((added++))
            else
                ((created++))
            fi
        else
            ((skipped++))
        fi
        echo "---"
    done < <(find "$directory" -maxdepth 1 -type f -print0)
    
    if [ "$ADD_TO_TRANSMISSION" = true ]; then
        added_to_transmission=$added
    else
        created_torrents=$created
    fi
    skipped_files=$skipped
}

# Счетчики для статистики
total_files=0
processed_files=0
created_torrents=0
added_to_transmission=0
skipped_files=0

echo "🚀 Начинаем обработку..."
echo "=========================================="

# Обрабатываем в зависимости от типа цели и режима
if [ "$TARGET_TYPE" = "file" ]; then
    total_files=1
    if process_single_file "$TARGET_PATH"; then
        processed_files=1
        if [ "$ADD_TO_TRANSMISSION" = true ]; then
            added_to_transmission=1
        else
            created_torrents=1
        fi
    else
        skipped_files=1
    fi
else
    # Для директории
    if [ "$DIR_TORRENT" = true ]; then
        # Обрабатываем директорию как единый торрент
        total_files=1
        if process_directory_as_torrent "$TARGET_DIR"; then
            processed_files=1
            if [ "$ADD_TO_TRANSMISSION" = true ]; then
                added_to_transmission=1
            else
                created_torrents=1
            fi
        else
            skipped_files=1
        fi
    else
        # Обрабатываем каждый файл в директории отдельно
        total_files=$(find "$TARGET_DIR" -maxdepth 1 -type f | wc -l)
        process_directory_files "$TARGET_DIR"
        if [ "$ADD_TO_TRANSMISSION" = true ]; then
            processed_files=$added_to_transmission
        else
            processed_files=$created_torrents
        fi
    fi
fi

# Выводим итоговую статистику
echo ""
echo "📊 ИТОГОВАЯ СТАТИСТИКА:"
echo "======================"
echo "🎯 Обработанный объект: $TARGET_PATH ($TARGET_TYPE)"

if [ "$DIR_TORRENT" = true ] && [ "$TARGET_TYPE" = "directory" ]; then
    echo "📦 Режим: директория как единый торрент"
    echo "🏷️  Имя торрента: $TORRENT_NAME"
else
    echo "📁 Режим: отдельные файлы"
fi

echo "📁 Всего файлов/объектов: $total_files"

if [ "$ADD_TO_TRANSMISSION" = true ]; then
    echo "✅ Успешно обработано: $processed_files"
    echo "🔨 Создано новых торрентов: $created_torrents"
    echo "📤 Добавлено в transmission: $added_to_transmission"
else
    echo "✅ Создано торрентов: $processed_files"
    echo "💾 Торренты сохранены в: $TORRENT_DIR"
fi

echo "⏭️  Пропущено: $skipped_files"
echo ""

if [ "$ADD_TO_TRANSMISSION" = true ] && [ $processed_files -gt 0 ]; then
    echo "📋 Текущий статус transmission:"
    echo "=============================="
    if [ -n "$TRANSMISSION_USER" ] && [ -n "$TRANSMISSION_PASSWORD" ]; then
        transmission-remote "$TRANSMISSION_HOST:$TRANSMISSION_PORT" \
            --auth "$TRANSMISSION_USER:$TRANSMISSION_PASSWORD" \
            -l
    else
        transmission-remote "$TRANSMISSION_HOST:$TRANSMISSION_PORT" \
            -l
    fi
fi

echo ""
echo "✅ Обработка завершена!"
